/*
 * Copyright (C) 2016 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

//******************************************************************************
// Version history:
// 1.0 07/17             Initial version. (Nima Eskandari)
// 1.1 07/17             Added Comments. (Nima Eskandari)
//----------------------------------------------------------------------------
//   Designed 2017 by Texas Instruments
//
//   Nima Eskandari
//   Texas Instruments Inc.
//   August 2017
//   Built with CCS Version: Code Composer Studio v7
//******************************************************************************

#include <stdio.h>
#include <stdbool.h>
#include <stddef.h>
#include <unistd.h>
#include <string.h>

#include <Board.h>
#include <software_uart.h>

#include <ti/sysbios/BIOS.h>
#include <xdc/runtime/Types.h>
#include <ti/drivers/GPIO.h>

#include <ti/drivers/timer/GPTimerCC26XX.h>
#include <semaphore.h>

GPTimerCC26XX_Handle uartDelayTimer;
GPTimerCC26XX_Params uartTimerParams;
sem_t uartTimerSemaphore;

void delayOneBit();

void uartTimerCallback(GPTimerCC26XX_Handle handle, GPTimerCC26XX_IntMask interruptMask)
{
    sem_post(&uartTimerSemaphore);
}

void InitializeSWUartDelayTimer()
{
    sem_init(&uartTimerSemaphore, 0, 0);
    GPTimerCC26XX_Params_init(&uartTimerParams);
    uartTimerParams.width          = GPT_CONFIG_16BIT;
    uartTimerParams.mode           = GPT_MODE_ONESHOT_UP;
    uartTimerParams.debugStallMode = GPTimerCC26XX_DEBUG_STALL_OFF;
}

void delayOneBit()
{
    uartDelayTimer = GPTimerCC26XX_open(Board_GPTIMER1A, &uartTimerParams);
    if(uartDelayTimer == NULL) {

    }
    GPTimerCC26XX_Value loadVal = (48 * 10) - 1 ;
    GPTimerCC26XX_setLoadValue(uartDelayTimer, loadVal);
    GPTimerCC26XX_registerInterrupt(uartDelayTimer, uartTimerCallback, GPT_INT_TIMEOUT);

    GPTimerCC26XX_start(uartDelayTimer);
    sem_wait(&uartTimerSemaphore);
    GPTimerCC26XX_unregisterInterrupt(uartDelayTimer);
    GPTimerCC26XX_stop(uartDelayTimer);
    GPTimerCC26XX_close(uartDelayTimer);
}


void SoftwareUartPrintString(uint8_t * array)
{
    int i = 0;
    for (i = 0; i < strlen((const char *)array); i++)
    {
        SoftwareUartPrintChar(array[i]);
    }
}

void SoftwareUartPrintChar(uint8_t data)
{
    int cnt = 0;
    // start bit
    GPIO_write(Board_SW_UART_TX, 0);
    delayOneBit();

    //data
    for(cnt = 0 ; cnt < 8 ; cnt++)
    {
        if((data >> cnt) & 0x01)
        {
            GPIO_write(Board_SW_UART_TX, 1);
        }
        else
        {
            GPIO_write(Board_SW_UART_TX, 0);
        }
        delayOneBit();
    }

    //stop bit
    GPIO_write(Board_SW_UART_TX, 1);
    delayOneBit();
}

